/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.server.plugins;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Ordering;
import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.annotation.CheckForNull;
import org.picocontainer.Startable;
import org.sonar.api.Plugin;
import org.sonar.api.SonarRuntime;
import org.sonar.api.utils.MessageException;
import org.sonar.api.utils.log.Logger;
import org.sonar.api.utils.log.Loggers;
import org.sonar.core.platform.PluginInfo;
import org.sonar.core.platform.PluginLoader;
import org.sonar.core.platform.PluginRepository;
import org.sonar.core.util.FileUtils;
import org.sonar.server.platform.ServerFileSystem;
import org.sonar.updatecenter.common.Version;

public class ServerPluginRepository
implements PluginRepository,
Startable {
    private static final Logger LOG = Loggers.get(ServerPluginRepository.class);
    private static final String[] JAR_FILE_EXTENSIONS = new String[]{"jar"};
    private static final Set<String> DEFAULT_BLACKLISTED_PLUGINS = ImmutableSet.of((Object)"scmactivity", (Object)"issuesreport", (Object)"genericcoverage");
    private static final Set<String> FORBIDDEN_COMPATIBLE_PLUGINS = ImmutableSet.of((Object)"sqale", (Object)"report", (Object)"views");
    private static final Joiner SLASH_JOINER = Joiner.on((String)" / ").skipNulls();
    private static final String NOT_STARTED_YET = "not started yet";
    private final SonarRuntime runtime;
    private final ServerFileSystem fs;
    private final PluginLoader loader;
    private final AtomicBoolean started = new AtomicBoolean(false);
    private Set<String> blacklistedPluginKeys = DEFAULT_BLACKLISTED_PLUGINS;
    private final Map<String, PluginInfo> pluginInfosByKeys = new HashMap<String, PluginInfo>();
    private final Map<String, Plugin> pluginInstancesByKeys = new HashMap<String, Plugin>();
    private final Map<ClassLoader, String> keysByClassLoader = new HashMap<ClassLoader, String>();

    public ServerPluginRepository(SonarRuntime runtime, ServerFileSystem fs, PluginLoader loader) {
        this.runtime = runtime;
        this.fs = fs;
        this.loader = loader;
    }

    @VisibleForTesting
    void setBlacklistedPluginKeys(Set<String> keys) {
        this.blacklistedPluginKeys = keys;
    }

    public void start() {
        this.loadPreInstalledPlugins();
        this.moveDownloadedPlugins();
        this.unloadIncompatiblePlugins();
        this.logInstalledPlugins();
        this.loadInstances();
        this.started.set(true);
    }

    public void stop() {
        this.loader.unload(this.pluginInstancesByKeys.values());
        this.pluginInstancesByKeys.clear();
        this.pluginInfosByKeys.clear();
        this.keysByClassLoader.clear();
        this.started.set(true);
    }

    @CheckForNull
    public String getPluginKey(Object extension) {
        return this.keysByClassLoader.get(extension.getClass().getClassLoader());
    }

    private void loadPreInstalledPlugins() {
        for (File file : ServerPluginRepository.listJarFiles(this.fs.getInstalledPluginsDir())) {
            PluginInfo info = PluginInfo.create((File)file);
            this.registerPluginInfo(info);
        }
    }

    private void moveDownloadedPlugins() {
        if (this.fs.getDownloadedPluginsDir().exists()) {
            for (File sourceFile : ServerPluginRepository.listJarFiles(this.fs.getDownloadedPluginsDir())) {
                this.overrideAndRegisterPlugin(sourceFile);
            }
        }
    }

    private void registerPluginInfo(PluginInfo info) {
        String pluginKey = info.getKey();
        if (this.blacklistedPluginKeys.contains(pluginKey)) {
            LOG.warn("Plugin {} [{}] is blacklisted and is being uninstalled", (Object)info.getName(), (Object)pluginKey);
            FileUtils.deleteQuietly((File)info.getNonNullJarFile());
            return;
        }
        if (FORBIDDEN_COMPATIBLE_PLUGINS.contains(pluginKey)) {
            throw MessageException.of((String)String.format("Plugin '%s' is no longer compatible with this version of SonarQube", pluginKey));
        }
        PluginInfo existing = this.pluginInfosByKeys.put(pluginKey, info);
        if (existing != null) {
            throw MessageException.of((String)String.format("Found two versions of the plugin %s [%s] in the directory extensions/plugins. Please remove one of %s or %s.", info.getName(), pluginKey, info.getNonNullJarFile().getName(), existing.getNonNullJarFile().getName()));
        }
    }

    private void overrideAndRegisterPlugin(File sourceFile) {
        File destDir = this.fs.getInstalledPluginsDir();
        File destFile = new File(destDir, sourceFile.getName());
        if (destFile.exists()) {
            FileUtils.deleteQuietly((File)destFile);
        }
        try {
            org.apache.commons.io.FileUtils.moveFile((File)sourceFile, (File)destFile);
        }
        catch (IOException e) {
            throw new IllegalStateException(String.format("Fail to move plugin: %s to %s", sourceFile.getAbsolutePath(), destFile.getAbsolutePath()), e);
        }
        PluginInfo info = PluginInfo.create((File)destFile);
        PluginInfo existing = this.pluginInfosByKeys.put(info.getKey(), info);
        if (existing != null) {
            if (!existing.getNonNullJarFile().getName().equals(destFile.getName())) {
                FileUtils.deleteQuietly((File)existing.getNonNullJarFile());
            }
            LOG.info("Plugin {} [{}] updated to version {}", new Object[]{info.getName(), info.getKey(), info.getVersion()});
        } else {
            LOG.info("Plugin {} [{}] installed", (Object)info.getName(), (Object)info.getKey());
        }
    }

    private void unloadIncompatiblePlugins() {
        HashSet<String> removedKeys = new HashSet<String>();
        do {
            removedKeys.clear();
            for (PluginInfo plugin : this.pluginInfosByKeys.values()) {
                if (ServerPluginRepository.isCompatible(plugin, this.runtime, this.pluginInfosByKeys)) continue;
                removedKeys.add(plugin.getKey());
            }
            for (String removedKey : removedKeys) {
                this.pluginInfosByKeys.remove(removedKey);
            }
        } while (!removedKeys.isEmpty());
    }

    @VisibleForTesting
    static boolean isCompatible(PluginInfo plugin, SonarRuntime runtime, Map<String, PluginInfo> allPluginsByKeys) {
        if (Strings.isNullOrEmpty((String)plugin.getMainClass()) && Strings.isNullOrEmpty((String)plugin.getBasePlugin())) {
            LOG.warn("Plugin {} [{}] is ignored because entry point class is not defined", (Object)plugin.getName(), (Object)plugin.getKey());
            return false;
        }
        if (!plugin.isCompatibleWith(runtime.getApiVersion().toString())) {
            throw MessageException.of((String)String.format("Plugin %s [%s] requires at least SonarQube %s", plugin.getName(), plugin.getKey(), plugin.getMinimalSqVersion()));
        }
        if (!Strings.isNullOrEmpty((String)plugin.getBasePlugin()) && !allPluginsByKeys.containsKey(plugin.getBasePlugin())) {
            LOG.warn("Plugin {} [{}] is ignored because its base plugin [{}] is not installed", new Object[]{plugin.getName(), plugin.getKey(), plugin.getBasePlugin()});
            return false;
        }
        for (PluginInfo.RequiredPlugin requiredPlugin : plugin.getRequiredPlugins()) {
            PluginInfo installedRequirement = allPluginsByKeys.get(requiredPlugin.getKey());
            if (installedRequirement == null) {
                LOG.warn("Plugin {} [{}] is ignored because the required plugin [{}] is not installed", new Object[]{plugin.getName(), plugin.getKey(), requiredPlugin.getKey()});
                return false;
            }
            Version installedRequirementVersion = installedRequirement.getVersion();
            if (installedRequirementVersion == null || requiredPlugin.getMinimalVersion().compareToIgnoreQualifier(installedRequirementVersion) <= 0) continue;
            LOG.warn("Plugin {} [{}] is ignored because the version {}\u00a0of required plugin [{}] is not supported", new Object[]{plugin.getName(), plugin.getKey(), requiredPlugin.getKey(), requiredPlugin.getMinimalVersion()});
            return false;
        }
        return true;
    }

    private void logInstalledPlugins() {
        List orderedPlugins = Ordering.natural().sortedCopy(this.pluginInfosByKeys.values());
        for (PluginInfo plugin : orderedPlugins) {
            LOG.info("Deploy plugin {}", (Object)SLASH_JOINER.join((Object)plugin.getName(), (Object)plugin.getVersion(), new Object[]{plugin.getImplementationBuild()}));
        }
    }

    private void loadInstances() {
        this.pluginInstancesByKeys.putAll(this.loader.load(this.pluginInfosByKeys));
        for (Map.Entry<String, Plugin> e : this.pluginInstancesByKeys.entrySet()) {
            this.keysByClassLoader.put(e.getValue().getClass().getClassLoader(), e.getKey());
        }
    }

    public void uninstall(String pluginKey, File uninstallDir) {
        HashSet<String> uninstallKeys = new HashSet<String>();
        uninstallKeys.add(pluginKey);
        this.appendDependentPluginKeys(pluginKey, uninstallKeys);
        for (String uninstallKey : uninstallKeys) {
            PluginInfo info = this.getPluginInfo(uninstallKey);
            try {
                if (!this.getPluginFile(info).exists()) {
                    LOG.info("Plugin already uninstalled: {} [{}]", (Object)info.getName(), (Object)info.getKey());
                    continue;
                }
                LOG.info("Uninstalling plugin {} [{}]", (Object)info.getName(), (Object)info.getKey());
                File masterFile = this.getPluginFile(info);
                org.apache.commons.io.FileUtils.moveFileToDirectory((File)masterFile, (File)uninstallDir, (boolean)true);
            }
            catch (IOException e) {
                throw new IllegalStateException(String.format("Fail to uninstall plugin %s [%s]", info.getName(), info.getKey()), e);
            }
        }
    }

    public void cancelUninstalls(File uninstallDir) {
        for (File file : ServerPluginRepository.listJarFiles(uninstallDir)) {
            try {
                org.apache.commons.io.FileUtils.moveFileToDirectory((File)file, (File)this.fs.getInstalledPluginsDir(), (boolean)false);
            }
            catch (IOException e) {
                throw new IllegalStateException("Fail to cancel plugin uninstalls", e);
            }
        }
    }

    private void appendDependentPluginKeys(String pluginKey, Set<String> appendTo) {
        for (PluginInfo otherPlugin : this.getPluginInfos()) {
            if (otherPlugin.getKey().equals(pluginKey)) continue;
            for (PluginInfo.RequiredPlugin requirement : otherPlugin.getRequiredPlugins()) {
                if (!requirement.getKey().equals(pluginKey)) continue;
                appendTo.add(otherPlugin.getKey());
                this.appendDependentPluginKeys(otherPlugin.getKey(), appendTo);
            }
        }
    }

    private File getPluginFile(PluginInfo info) {
        return new File(this.fs.getInstalledPluginsDir(), info.getNonNullJarFile().getName());
    }

    public Map<String, PluginInfo> getPluginInfosByKeys() {
        return this.pluginInfosByKeys;
    }

    public Collection<PluginInfo> getPluginInfos() {
        Preconditions.checkState((boolean)this.started.get(), (Object)NOT_STARTED_YET);
        return ImmutableList.copyOf(this.pluginInfosByKeys.values());
    }

    public PluginInfo getPluginInfo(String key) {
        Preconditions.checkState((boolean)this.started.get(), (Object)NOT_STARTED_YET);
        PluginInfo info = this.pluginInfosByKeys.get(key);
        if (info == null) {
            throw new IllegalArgumentException(String.format("Plugin [%s] does not exist", key));
        }
        return info;
    }

    public Plugin getPluginInstance(String key) {
        Preconditions.checkState((boolean)this.started.get(), (Object)NOT_STARTED_YET);
        Plugin plugin = this.pluginInstancesByKeys.get(key);
        Preconditions.checkArgument((plugin != null ? 1 : 0) != 0, (String)"Plugin [%s] does not exist", (Object[])new Object[]{key});
        return plugin;
    }

    public boolean hasPlugin(String key) {
        Preconditions.checkState((boolean)this.started.get(), (Object)NOT_STARTED_YET);
        return this.pluginInfosByKeys.containsKey(key);
    }

    private static Collection<File> listJarFiles(File dir) {
        if (dir.exists()) {
            return org.apache.commons.io.FileUtils.listFiles((File)dir, (String[])JAR_FILE_EXTENSIONS, (boolean)false);
        }
        return Collections.emptyList();
    }
}

