/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.server.authentication;

import com.google.common.base.Preconditions;
import java.security.SecureRandom;
import java.util.Objects;
import javax.annotation.Nullable;
import org.apache.commons.codec.digest.DigestUtils;
import org.mindrot.jbcrypt.BCrypt;
import org.sonar.db.DbClient;
import org.sonar.db.DbSession;
import org.sonar.db.user.UserDto;
import org.sonar.server.authentication.event.AuthenticationEvent;
import org.sonar.server.authentication.event.AuthenticationException;

public class CredentialsLocalAuthentication {
    private final DbClient dbClient;
    private static final SecureRandom SECURE_RANDOM = new SecureRandom();
    private static final HashMethod DEFAULT = HashMethod.BCRYPT;

    public CredentialsLocalAuthentication(DbClient dbClient) {
        this.dbClient = dbClient;
    }

    public void authenticate(DbSession session, UserDto user, @Nullable String password, AuthenticationEvent.Method method) {
        HashMethod hashMethod;
        if (user.getHashMethod() == null) {
            throw AuthenticationException.newBuilder().setSource(AuthenticationEvent.Source.local(method)).setLogin(user.getLogin()).setMessage("null hash method").build();
        }
        try {
            hashMethod = HashMethod.valueOf(user.getHashMethod());
        }
        catch (IllegalArgumentException ex) {
            throw AuthenticationException.newBuilder().setSource(AuthenticationEvent.Source.local(method)).setLogin(user.getLogin()).setMessage(String.format("Unknown hash method [%s]", user.getHashMethod())).build();
        }
        AuthenticationResult result = hashMethod.checkCredentials(user, password);
        if (!result.isSuccessful()) {
            throw AuthenticationException.newBuilder().setSource(AuthenticationEvent.Source.local(method)).setLogin(user.getLogin()).setMessage(result.getFailureMessage()).build();
        }
        if (hashMethod != DEFAULT) {
            DEFAULT.storeHashPassword(user, password);
            this.dbClient.userDao().update(session, user);
        }
    }

    public void storeHashPassword(UserDto user, String password) {
        DEFAULT.storeHashPassword(user, password);
    }

    private static final class BcryptFunction
    implements HashFunction {
        private BcryptFunction() {
        }

        @Override
        public AuthenticationResult checkCredentials(UserDto user, String password) {
            if (!BCrypt.checkpw((String)password, (String)user.getCryptedPassword())) {
                return new AuthenticationResult(false, "wrong password");
            }
            return new AuthenticationResult(true, "");
        }

        @Override
        public void storeHashPassword(UserDto user, String password) {
            Objects.requireNonNull(password, "Password cannot be null");
            user.setHashMethod(HashMethod.BCRYPT.name()).setCryptedPassword(BCrypt.hashpw((String)password, (String)BCrypt.gensalt((int)12))).setSalt(null);
        }
    }

    private static final class Sha1Function
    implements HashFunction {
        private Sha1Function() {
        }

        @Override
        public AuthenticationResult checkCredentials(UserDto user, String password) {
            if (user.getCryptedPassword() == null) {
                return new AuthenticationResult(false, "null password in DB");
            }
            if (user.getSalt() == null) {
                return new AuthenticationResult(false, "null salt");
            }
            if (!user.getCryptedPassword().equals(Sha1Function.hash(user.getSalt(), password))) {
                return new AuthenticationResult(false, "wrong password");
            }
            return new AuthenticationResult(true, "");
        }

        @Override
        public void storeHashPassword(UserDto user, String password) {
            Objects.requireNonNull(password, "Password cannot be null");
            byte[] saltRandom = new byte[20];
            SECURE_RANDOM.nextBytes(saltRandom);
            String salt = DigestUtils.sha1Hex((byte[])saltRandom);
            user.setHashMethod(HashMethod.SHA1.name()).setCryptedPassword(Sha1Function.hash(salt, password)).setSalt(salt);
        }

        private static String hash(String salt, String password) {
            return DigestUtils.sha1Hex((String)("--" + salt + "--" + password + "--"));
        }
    }

    public static interface HashFunction {
        public AuthenticationResult checkCredentials(UserDto var1, String var2);

        public void storeHashPassword(UserDto var1, String var2);
    }

    private static class AuthenticationResult {
        private final boolean successful;
        private final String failureMessage;

        private AuthenticationResult(boolean successful, String failureMessage) {
            Preconditions.checkArgument((successful && failureMessage.isEmpty() || !successful && !failureMessage.isEmpty() ? 1 : 0) != 0, (Object)"Incorrect parameters");
            this.successful = successful;
            this.failureMessage = failureMessage;
        }

        public boolean isSuccessful() {
            return this.successful;
        }

        public String getFailureMessage() {
            return this.failureMessage;
        }
    }

    public static enum HashMethod implements HashFunction
    {
        SHA1(new Sha1Function()),
        BCRYPT(new BcryptFunction());

        private HashFunction hashFunction;

        private HashMethod(HashFunction hashFunction) {
            this.hashFunction = hashFunction;
        }

        @Override
        public AuthenticationResult checkCredentials(UserDto user, String password) {
            return this.hashFunction.checkCredentials(user, password);
        }

        @Override
        public void storeHashPassword(UserDto user, String password) {
            this.hashFunction.storeHashPassword(user, password);
        }
    }
}

